
export async function callLongcatApi(prompt, systemPrompt, taskType, apiKey, model, onLog = () => { }, signal = null) {
    if (!apiKey) {
        throw new Error("Longcat API Key is missing.");
    }

    onLog(`🔑 Using Longcat AI...`, 'info');

    const url = "https://api.longcat.chat/openai/v1/chat/completions";

    const messages = [];
    if (systemPrompt) {
        messages.push({ role: "system", content: systemPrompt });
    }
    messages.push({ role: "user", content: prompt });

    const payload = {
        model: model || "LongCat-Flash-Chat", // Default model
        messages: messages,
        temperature: 0.7
        // max_tokens: 4000 // Let's leave it to default or high if needed
    };

    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), 180000); // 3 min timeout for safety

    const onAbort = () => controller.abort();
    if (signal) signal.addEventListener('abort', onAbort);

    try {
        const response = await fetch(url, {
            method: "POST",
            headers: {
                "Authorization": `Bearer ${apiKey}`,
                "Content-Type": "application/json"
            },
            body: JSON.stringify(payload),
            signal: controller.signal
        });

        clearTimeout(timeoutId);
        if (signal) signal.removeEventListener('abort', onAbort);

        const data = await response.json();

        if (!response.ok) {
            const errorMsg = data.error?.message || response.statusText;
            onLog(`Longcat API Error ${response.status}: ${errorMsg}`, 'error');
            throw new Error(`Longcat API Error: ${errorMsg}`);
        }

        if (data.choices && data.choices[0] && data.choices[0].message) {
            onLog(`✅ Successfully generated ${taskType} (Longcat)`, 'success');
            return data.choices[0].message.content;
        } else {
            onLog(`Invalid Longcat response structure`, 'error');
            throw new Error("Invalid Longcat response structure");
        }

    } catch (e) {
        clearTimeout(timeoutId);
        if (signal) signal.removeEventListener('abort', onAbort);

        if (e.name === 'AbortError') {
            if (signal?.aborted) throw new Error("Aborted by user");
            onLog(`Request timed out (Longcat)`, 'error');
        } else {
            onLog(`Longcat Network/API Exception: ${e.toString()}`, 'error');
        }
        throw e;
    }
}
